/*
 * SPDX-License-Identifier: BSD-3-Clause
 * SPDX-FileCopyrightText: Copyright TF-RMM Contributors.
 */

#ifndef SMC_RSI_H
#define SMC_RSI_H

#include <smc.h>
#include <stddef.h>
#include <utils_def.h>

/*
 * This file describes the Realm Services Interface (RSI) Application Binary
 * Interface (ABI) for SMC calls made from within the Realm to the RMM and
 * serviced by the RMM.
 *
 * See doc/rmm_interface.md for more details.
 */

/*
 * The major version number of the RSI implementation.  Increase this whenever
 * the binary format or semantics of the SMC calls change.
 */
#define RSI_ABI_VERSION_MAJOR		12U

/*
 * The minor version number of the RSI implementation.  Increase this when
 * a bug is fixed, or a feature is added without breaking binary compatibility.
 */
#define RSI_ABI_VERSION_MINOR		0

#define RSI_ABI_VERSION			((RSI_ABI_VERSION_MAJOR << 16U) | \
					 RSI_ABI_VERSION_MINOR)

#define RSI_ABI_VERSION_GET_MAJOR(_version) ((_version) >> 16U)
#define RSI_ABI_VERSION_GET_MINOR(_version) ((_version) & 0xFFFFU)

#define IS_SMC64_RSI_FID(_fid)		IS_SMC64_STD_FAST_IN_RANGE(RSI, _fid)

#define SMC64_RSI_FID(_offset)		SMC64_STD_FID(RSI, _offset)

#define SMC_RSI_ABI_VERSION		SMC64_RSI_FID(U(0x0))

/* RSI Status code enumeration as per Section D4.3.6 of the RMM Spec */
typedef enum {
	/* Command completed successfully */
	RSI_SUCCESS = 0U,

	/*
	 * The value of a command input value
	 * caused the command to fail
	 */
	RSI_ERROR_INPUT	= 1U,

	/*
	 * The state of the current Realm or current REC
	 * does not match the state expected by the command
	 */
	RSI_ERROR_STATE	= 2U,

	/* The operation requested by the command is not complete */
	RSI_INCOMPLETE = 3U,

	RSI_ERROR_COUNT
} rsi_status_t;

/*
 * Returns a measurement.
 * arg1: Measurement index (0..4), measurement (RIM or REM) to read
 * ret0: Status / error
 * ret1: Measurement value, bytes:  0 -  7
 * ret2: Measurement value, bytes:  7 - 15
 * ret3: Measurement value, bytes: 16 - 23
 * ret4: Measurement value, bytes: 24 - 31
 * ret5: Measurement value, bytes: 32 - 39
 * ret6: Measurement value, bytes: 40 - 47
 * ret7: Measurement value, bytes: 48 - 55
 * ret8: Measurement value, bytes: 56 - 63
 */
#define SMC_RSI_MEASUREMENT_READ	SMC64_RSI_FID(U(0x2))

/*
 * Extends a REM.
 * arg0:  Measurement index (1..4), measurement (REM) to extend
 * arg1:  Measurement size in bytes
 * arg3:  Challenge value, bytes:  0 -  7
 * arg4:  Challenge value, bytes:  7 - 15
 * arg5:  Challenge value, bytes: 16 - 23
 * arg6:  Challenge value, bytes: 24 - 31
 * arg7:  Challenge value, bytes: 32 - 39
 * arg8:  Challenge value, bytes: 40 - 47
 * arg9:  Challenge value, bytes: 48 - 55
 * arg10: Challenge value, bytes: 56 - 63
 * ret0:  Status / error
 */
#define SMC_RSI_MEASUREMENT_EXTEND	SMC64_RSI_FID(U(0x3))

/*
 * Initialize the operation to retrieve an attestation token.
 * arg1: The IPA of token buffer
 * arg2: Challenge value, bytes:  0 -  7
 * arg3: Challenge value, bytes:  7 - 15
 * arg4: Challenge value, bytes: 16 - 23
 * arg5: Challenge value, bytes: 24 - 31
 * arg6: Challenge value, bytes: 32 - 39
 * arg7: Challenge value, bytes: 40 - 47
 * arg8: Challenge value, bytes: 48 - 55
 * arg9: Challenge value, bytes: 56 - 63
 * ret0: Status / error
 * ret1: Size of completed token in bytes
 */
#define SMC_RSI_ATTEST_TOKEN_INIT	SMC64_RSI_FID(U(0x4))

/*
 * Continue the operation to retrieve an attestation token.
 * arg1: The IPA of token buffer
 * ret0: Status / error
 * ret1: Size of completed token in bytes
 */
#define SMC_RSI_ATTEST_TOKEN_CONTINUE	SMC64_RSI_FID(U(0x5))

struct rsi_realm_config {
	/* IPA width in bits */
	SET_MEMBER(unsigned long ipa_width, 0, 0x1000);	/* Offset 0 */
};

COMPILER_ASSERT(sizeof(struct rsi_realm_config) == 0x1000);

COMPILER_ASSERT(offsetof(struct rsi_realm_config, ipa_width) == 0);

/*
 * arg0 == struct rsi_realm_config address
 */
#define SMC_RSI_REALM_CONFIG		SMC64_RSI_FID(U(0x6))

/*
 * arg0 == IPA address of target region
 * arg1 == Size of target region in bytes
 * arg2 == RIPAS value
 * ret0 == Status / error
 * ret1 == Top of modified IPA range
 */
#define SMC_RSI_IPA_STATE_SET		SMC64_RSI_FID(U(0x7))

/*
 * arg0 == IPA
 * ret0 == Status / error
 * ret1 == RIPAS value
 */
#define SMC_RSI_IPA_STATE_GET		SMC64_RSI_FID(U(0x8))

#define RSI_HOST_CALL_NR_GPRS		7U

struct rsi_host_call {
	SET_MEMBER(struct {
		/* Immediate value */
		unsigned int imm;		/* Offset 0 */
		/* Registers */
		unsigned long gprs[RSI_HOST_CALL_NR_GPRS];
		}, 0, 0x100);
};

COMPILER_ASSERT(sizeof(struct rsi_host_call) == 0x100);

COMPILER_ASSERT(offsetof(struct rsi_host_call, imm) == 0);
COMPILER_ASSERT(offsetof(struct rsi_host_call, gprs[0]) == 8);
COMPILER_ASSERT(offsetof(struct rsi_host_call,
			 gprs[RSI_HOST_CALL_NR_GPRS - 1]) ==
			 8 * RSI_HOST_CALL_NR_GPRS);

/*
 * arg0 == struct rsi_host_call addr
 */
#define SMC_RSI_HOST_CALL		SMC64_RSI_FID(U(0x9))

#endif /* SMC_RSI_H */
