/*
 * SPDX-License-Identifier: BSD-3-Clause
 * SPDX-FileCopyrightText: Copyright TF-RMM Contributors.
 */

#ifndef ATTESTATION_PRIV_H
#define ATTESTATION_PRIV_H

/*
 * A structure holding the context for generating a pseudo-random number derived
 * from a real random seed.
 */
struct attest_rng_context {
	int (*f_rng)(void *p_rng, unsigned char *output, size_t out_len);
	void *p_rng;
};

/*
 * Copy the rng_ctx for the current CPU to rng_ctx
 *
 * Arguments:
 * rng_ctx - Pointer to the target context structure
 */
void attest_get_cpu_rng_context(struct attest_rng_context *rng_ctx);

/*
 * Get a pointer to the keypair for signing realm attestation token.
 *
 * Arguments:
 * keypair - Pointer to the keypair for signing token.

 * Returns 0 on success, negative error code on error.
 */
int attest_get_realm_signing_key(const void **keypair);

/*
 * Query the attestation private key from monitor and generate the public
 * key by using MbedCryto lib. The key is cached internally for future
 * use. The function returns early if the key has been initialized.
 *
 * FPU context must be saved and FPU access should be enabled by caller.
 *
 * Returns 0 on success, negative error code on error.
 */
int attest_init_realm_attestation_key(void);

/*
 * Get the hash of the realm attestation public key. The public key hash is the
 * challenge value in the platform attestation token.
 *
 * Arguments:
 * public_key_hash - Get the buffer address and size which holds
 *                   the hash of the realm attestation public key.
 *
 * Returns 0 on success, negative error code on error.
 *
 */
int attest_get_realm_public_key_hash(struct q_useful_buf_c *public_key_hash);

/*
 * Get the realm attestation public key hash. The public key hash is the
 * challenge value in the platform attestation token.
 *
 * Arguments:
 * public_key - Get the buffer address and size which holds the realm
 *              attestation public key.
 *
 * Returns 0 on success, negative error code on error.
 */
int attest_get_realm_public_key(struct q_useful_buf_c *public_key);

/*
 * Get the platform token from monitor. This function needs to be called
 * after the Realm attestation key has been initialized.
 *
 * Returns 0 on success, negative error code on error.
 */
int attest_setup_platform_token(void);

/*
 * Get the hash algorithm to use for computing the hash of the realm public key.
 */
enum hash_algo attest_get_realm_public_key_hash_algo_id(void);

/*
 * Initialise PRNGs for all the CPUs
 *
 * FPU context must be saved and FPU access should be enabled by caller.
 *
 * Returns 0 on success, negative error code otherwise.
 *
 * This function creates a separate PRNG object for all the CPUs. The PRNGs are
 * used by Mbed TLS when it needs random data. The PRNGs are seeded with values
 * generated by a temporary PRNG, which is in turn is seeded with a real random
 * value.
 */
int attest_rnd_prng_init(void);

#endif /* ATTESTATION_PRIV_H */
